package org.example.common.oss;

import com.alibaba.fastjson.JSONObject;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.MatchMode;
import com.aliyun.oss.model.PolicyConditions;
import lombok.extern.slf4j.Slf4j;
import org.example.common.constant.Constants;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.sql.Date;
import java.util.Map;

/**
 * @author liutao
 * @since 2024/4/7
 */
@Slf4j
@Service
@ConditionalOnProperty(name = "oss.enable", havingValue = "true")
public class OssServiceImpl implements OssService{


    private OSSClient ossClient;

    @Resource
    private OssProperties ossProperties;

    @PostConstruct
    public void init() {
        ossClient = new OSSClient(ossProperties.getEndpoint(), ossProperties.getAccessKeyId(), ossProperties.getSecretAccessKey());
    }


    @Override
    public String uploadObject(InputStream is, String filePath, boolean isHttps) {
        if (filePath.startsWith("/")) {
            filePath = filePath.substring(1);
        }
        String bucketName = ossProperties.getBucketName();
        ossClient.putObject(bucketName,filePath,is);
        String httpPrefix = isHttps ? Constants.HTTPS : Constants.HTTP;
        String endpoint = ossProperties.getEndpoint();
        if (!endpoint.endsWith("/")) {
            endpoint = endpoint + "/";
        }
        return httpPrefix + bucketName + "." + endpoint + filePath;
    }

    @Override
    public String uploadObject(InputStream is, String filePath) {
        return uploadObject(is,filePath,false);
    }

    @Override
    public Map<String, Object> getPostSignature(String key) {
        String host = Constants.HTTPS + ossProperties.getBucketName() + "." + ossProperties.getEndpoint();
        try {
            long expireTime = 3600;
            long expireEndTime = System.currentTimeMillis() + expireTime * 1000;
            Date expiration = new Date(expireEndTime);
            PolicyConditions policyConditions = new PolicyConditions();
            policyConditions.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, 1048576000);
            policyConditions.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY,key);

            String postPolicy = ossClient.generatePostPolicy(expiration, policyConditions);
            byte[] binaryData = postPolicy.getBytes(StandardCharsets.UTF_8);
            String encodedPolicy = BinaryUtil.toBase64String(binaryData);
            String postSignature = ossClient.calculatePostSignature(postPolicy);

            JSONObject respMap = new JSONObject();
            respMap.put("OSSAccessKeyId", ossProperties.getAccessKeyId());
            respMap.put("policy", encodedPolicy);
            respMap.put("signature", postSignature);
            respMap.put("host", host);
            respMap.put("expire", String.valueOf(expireEndTime / 1000));
            respMap.put("key",key);
            return  respMap;
        } catch (Exception e) {
            log.error("获取oss上传签名失败");
            log.error(e.getMessage(),e);
        }
        return null;
    }
}
