package org.example.common.util;

import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.HttpServletRequest;

/**
 * @author liutao
 * @since 2024/4/11
 */
@Slf4j
public class IpUtil {


    private static final String[] IP_HEADER_CANDIDATES = {
            "X-Forwarded-For",
            "Proxy-Client-IP",
            "WL-Proxy-Client-IP",
            "HTTP_CLIENT_IP",
            "HTTP_X_FORWARDED_FOR",
            "X-Real-IP"
    };

    public static String getIP() {
        HttpServletRequest request = WebUtil.getRequest();
        String ip = null;

        for (String header : IP_HEADER_CANDIDATES) {
            ip = request.getHeader(header);
            if (StringUtils.isNotBlank(ip) && !"unknown".equalsIgnoreCase(ip)) {
                // 检查IP地址格式
                if (!isValidIpAddress(ip)) {
                    log.warn("Invalid IP format for header: {}" , header);
                    // IP格式不正确，继续查找下一个头部
                    ip = null;
                    continue;
                }

                // 对于X-Forwarded-For头部，获取第一个IP地址
                if (header.equalsIgnoreCase("X-Forwarded-For")) {
                    ip = ip.split(",")[0];
                }
                // 找到有效的IP地址，跳出循环
                break;
            }
        }

        if (ip == null || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
            log.info("getRemoteAddr ip: {}" , ip);
        } else {
            log.info("Client IP from header: {}" , ip);
        }

        return ip;
    }



    public static boolean isValidIpAddress(String ipAddress) {
        if (ipAddress == null) {
            return false;
        }
        String[] parts = ipAddress.split("\\.");
        if (parts.length != 4) {
            return false;
        }
        for (String part : parts) {
            if (part.isEmpty() || part.length() > 3) {
                return false;
            }
            try {
                int num = Integer.parseInt(part);
                if (num < 0 || num > 255) {
                    return false;
                }
            } catch (NumberFormatException e) {
                return false;
            }
        }
        return true;
    }
}
