package org.example.common.cache;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.formula.functions.T;
import org.example.common.cache.conversion.TypeConversion;
import org.example.common.cache.data.RedisData;
import org.example.common.lock.DistributedLock;
import org.example.common.lock.factory.DistributedLockFactory;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * @author liutao
 * @since 2024/4/7
 */
@Slf4j
@Component
public class RedisCacheService {


    //缓存空数据的时长，单位秒
    private static final Long CACHE_NULL_TTL = 60L;
    //缓存的空数据
    private static final String EMPTY_VALUE = "";
    //缓存的空列表数据
    private static final String EMPTY_LIST_VALUE = "[]";
    //分布式锁key的后缀
    private static final String LOCK_SUFFIX = "_lock";
    //线程休眠的毫秒数
    private static final long THREAD_SLEEP_MILLISECONDS = 50;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private DistributedLockFactory distributedLockFactory;

    
    public void set(String key, Object value) {
        stringRedisTemplate.opsForValue().set(key, this.getValue(value));
    }

    
    public void set(String key, Object value, Long timeout, TimeUnit unit) {
        stringRedisTemplate.opsForValue().set(key, this.getValue(value), timeout, unit);
    }

    public void increment(String key) {
        stringRedisTemplate.opsForValue().increment(key);
    }

    public void increment(String key, Long delta) {
        stringRedisTemplate.opsForValue().increment(key, delta);
    }

    public void decrement(String key) {
        stringRedisTemplate.opsForValue().decrement(key);
    }

    public void decrement(String key, Long delta) {
        stringRedisTemplate.opsForValue().decrement(key, delta);
    }

    public void put(String key, String field, Object value) {
        stringRedisTemplate.opsForHash().put(key, field, this.getValue(value));
    }

    public void putAll(String key, Map<String, Object> map) {
        stringRedisTemplate.opsForHash().putAll(key, map);
    }

    public Object hGet(String key, String field) {
        return stringRedisTemplate.opsForHash().get(key, field);
    }

    public Integer hGetInteger(String key, String field) {
        Object object = stringRedisTemplate.opsForHash().get(key, field);
        if (object == null) {
            return null;
        }
        return Integer.parseInt(object.toString());
    }

    public Object hGetLong(String key, String field) {
        Object object = stringRedisTemplate.opsForHash().get(key, field);
        if (object == null) {
            return null;
        }
        return Long.parseLong(object.toString());
    }

    public <T> T hGet(String key, String field, Class<T> targetClass) {
        Object result = stringRedisTemplate.opsForHash().get(key, field);
        if (result == null) {
            return null;
        }
        try {
            return JSON.parseObject(result.toString(), targetClass);
        } catch (Exception e) {
            return null;
        }
    }

    public Map<Object, Object> hGetAll(String key) {
        return stringRedisTemplate.opsForHash().entries(key);
    }
    
    public Boolean expire(String key, long timeout, TimeUnit unit) {
        return stringRedisTemplate.expire(key, timeout, unit);
    }

    public Long addSet(String key, String... values) {
        return stringRedisTemplate.opsForSet().add(key, values);
    }
    public Boolean isMemberSet(String key, Object value) {
        return stringRedisTemplate.opsForSet().isMember(key, value.toString());
    }
    public Set<String> membersSet(String key) {
        return stringRedisTemplate.opsForSet().members(key);
    }

    public Long removeSet(String key, Object... values) {
        return stringRedisTemplate.opsForSet().remove(key, values);
    }

    public Long sizeSet(String key) {
        return stringRedisTemplate.opsForSet().size(key);
    }

    public String get(String key) {
        return stringRedisTemplate.opsForValue().get(key);
    }

    public Integer getInteger(String key) {
        String value = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isEmpty(value)) {
            return null;
        }
        return Integer.parseInt(value);
    }

    public Long getLong(String key) {
        String value = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isEmpty(value)) {
            return null;
        }
        return Long.parseLong(value);
    }
    
    public <T> T getObject(String key, Class<T> targetClass) {
        Object result = stringRedisTemplate.opsForValue().get(key);
        if (result == null) {
            return null;
        }
        try {
            return JSON.parseObject(result.toString(), targetClass);
        } catch (Exception e) {
            return null;
        }
    }

    
    public List<String> multiGet(Collection<String> keys) {
        return stringRedisTemplate.opsForValue().multiGet(keys);
    }

    
    public Set<String> keys(String pattern) {
        return stringRedisTemplate.keys(pattern);
    }

    
    public Boolean delete(String key) {
        if (StrUtil.isEmpty(key)) {
            return false;
        }
        return stringRedisTemplate.delete(key);
    }

    
    public <R, ID> R queryWithPassThrough(String keyPrefix, ID id, Class<R> type, Function<ID, R> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix, id);
        //从Redis查询缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        //缓存存在数据，直接返回
        if (StrUtil.isNotBlank(str)){
            //返回数据
            return this.getResult(str, type);
        }
        //缓存中存储的是空字符串
        if (str != null){
            //直接返回空
            return null;
        }
        //从数据库查询数据
        R r = dbFallback.apply(id);
        //数据数据为空
        if (r == null){
            stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
            return null;
        }
        //缓存数据
        this.set(key, r, timeout, unit);
        return r;
    }

    
    public <R> R queryWithPassThroughWithoutArgs(String keyPrefix, Class<R> type, Supplier<R> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix);
        //从Redis查询缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        //缓存存在数据，直接返回
        if (StrUtil.isNotBlank(str)){
            //返回数据
            return this.getResult(str, type);
        }
        //缓存中存储的是空字符串
        if (str != null){
            //直接返回空
            return null;
        }
        //从数据库查询数据
        R r = dbFallback.get();
        //数据数据为空
        if (r == null){
            stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
            return null;
        }
        //缓存数据
        this.set(key, r, timeout, unit);
        return r;
    }

    
    public <R, ID> List<R> queryWithPassThroughList(String keyPrefix, ID id, Class<R> type, Function<ID, List<R>> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix, id);
        //从Redis查询缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        //缓存存在数据，直接返回
        if (StrUtil.isNotBlank(str)){
            //返回数据
            return this.getResultList(str, type);
        }
        if (str != null){
            //直接返回数据
            return null;
        }
        List<R> r = dbFallback.apply(id);
        //数据库数据为空
        if (r == null || r.isEmpty()){
            stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
            return Collections.emptyList();
        }
        this.set(key, r, timeout, unit);
        return r;
    }

    
    public <R> List<R> queryWithPassThroughListWithoutArgs(String keyPrefix, Class<R> type, Supplier<List<R>> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix);
        //从Redis查询缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        //缓存存在数据，直接返回
        if (StrUtil.isNotBlank(str)){
            //返回数据
            return this.getResultList(str, type);
        }
        if (str != null){
            //直接返回数据
            return null;
        }
        List<R> r = dbFallback.get();
        //数据库数据为空
        if (r == null || r.isEmpty()){
            stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
            return null;
        }
        this.set(key, r, timeout, unit);
        return r;
    }




    
    public <R, ID> R queryWithMutex(String keyPrefix, ID id, Class<R> type, Function<ID, R> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix, id);
        //从Redis获取缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isNotBlank(str)){
            //存在数据，直接返回
            return this.getResult(str, type);
        }
        //缓存了空字符串
        if (str != null){
            return null;
        }
        String lockKey = this.getLockKey(key);
        R r = null;
        //获取分布式锁
        DistributedLock distributedLock = distributedLockFactory.getDistributedLock(lockKey);
        try {
            boolean isLock = distributedLock.tryLock();
            //获取分布式锁失败，重试
            if (!isLock){
                Thread.sleep(THREAD_SLEEP_MILLISECONDS);
                return queryWithMutex(keyPrefix, id, type, dbFallback, timeout, unit);
            }
            //获取锁成功, Double Check
            str = stringRedisTemplate.opsForValue().get(key);
            if (StrUtil.isNotBlank(str)){
                //存在数据，直接返回
                return this.getResult(str, type);
            }
            //成功获取到锁
            r = dbFallback.apply(id);
            //数据库本身不存在数据
            if (r == null){
                //缓存空数据
                this.set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
                return null;
            }
            //数据库存在数据
            this.set(key, r, timeout, unit);
        } catch (InterruptedException e) {
            log.error("query data with mutex |{}", e.getMessage());
            throw new RuntimeException(e);
        }finally {
            distributedLock.unlock();
        }
        return r;
    }

    
    public <R> R queryWithMutexWithoutArgs(String keyPrefix, Class<R> type, Supplier<R> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix);
        //从Redis获取缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isNotBlank(str)){
            //存在数据，直接返回
            return this.getResult(str, type);
        }
        //缓存了空字符串
        if (str != null){
            return null;
        }
        String lockKey = this.getLockKey(key);
        R r = null;
        //获取分布式锁
        DistributedLock distributedLock = distributedLockFactory.getDistributedLock(lockKey);
        try {
            boolean isLock = distributedLock.tryLock();
            //获取分布式锁失败，重试
            if (!isLock){
                Thread.sleep(THREAD_SLEEP_MILLISECONDS);
                return queryWithMutexWithoutArgs(keyPrefix, type, dbFallback, timeout, unit);
            }
            //获取锁成功, Double Check
            str = stringRedisTemplate.opsForValue().get(key);
            if (StrUtil.isNotBlank(str)){
                //存在数据，直接返回
                return this.getResult(str, type);
            }
            r = dbFallback.get();
            //数据库本身不存在数据
            if (r == null){
                //缓存空数据
                this.set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
                return null;
            }
            //数据库存在数据
            this.set(key, r, timeout, unit);
        } catch (InterruptedException e) {
            log.error("query data with mutex |{}", e.getMessage());
            throw new RuntimeException(e);
        }finally {
            distributedLock.unlock();
        }
        return r;
    }

    
    public <R, ID> List<R> queryWithMutexList(String keyPrefix, ID id, Class<R> type, Function<ID, List<R>> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix, id);
        //从Redis获取缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isNotBlank(str)){
            //存在数据，直接返回
            return this.getResultList(str, type);
        }
        //缓存了空字符串
        if (str != null){
            return null;
        }
        String lockKey = this.getLockKey(key);
        List<R> list = null;
        // 获取分布式锁
        DistributedLock distributedLock = distributedLockFactory.getDistributedLock(lockKey);
        try {
            boolean isLock = distributedLock.tryLock();
            //获取分布式锁失败，重试
            if (!isLock){
                Thread.sleep(THREAD_SLEEP_MILLISECONDS);
                //重试
                return queryWithMutexList(keyPrefix, id, type, dbFallback, timeout, unit);
            }
            //获取锁成功, Double Check
            str = stringRedisTemplate.opsForValue().get(key);
            if (StrUtil.isNotBlank(str)){
                //存在数据，直接返回
                return this.getResultList(str, type);
            }
            list = dbFallback.apply(id);
            //数据库本身不存在数据
            if (list == null){
                //缓存空数据
                stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
                return null;
            }
            //数据库存在数据
            this.set(key, list, timeout, unit);

        } catch (InterruptedException e) {
            log.error("query data with mutex list |{}", e.getMessage());
            throw new RuntimeException(e);
        }finally {
            distributedLock.unlock();
        }
        return list;
    }

    
    public <R> List<R> queryWithMutexListWithoutArgs(String keyPrefix, Class<R> type, Supplier<List<R>> dbFallback, Long timeout, TimeUnit unit) {
        //获取存储到Redis中的数据key
        String key = this.getKey(keyPrefix);
        //从Redis获取缓存数据
        String str = stringRedisTemplate.opsForValue().get(key);
        if (StrUtil.isNotBlank(str)){
            //存在数据，直接返回
            return this.getResultList(str, type);
        }
        //缓存了空字符串
        if (str != null){
            return null;
        }
        String lockKey = this.getLockKey(key);
        List<R> list = null;
        // 获取分布式锁
        DistributedLock distributedLock = distributedLockFactory.getDistributedLock(lockKey);
        try {
            boolean isLock = distributedLock.tryLock();
            //获取分布式锁失败，重试
            if (!isLock){
                Thread.sleep(THREAD_SLEEP_MILLISECONDS);
                //重试
                return queryWithMutexListWithoutArgs(keyPrefix, type, dbFallback, timeout, unit);
            }
            //获取锁成功, Double Check
            str = stringRedisTemplate.opsForValue().get(key);
            if (StrUtil.isNotBlank(str)){
                //存在数据，直接返回
                return this.getResultList(str, type);
            }
            list = dbFallback.get();
            //数据库本身不存在数据
            if (list == null){
                //缓存空数据
                stringRedisTemplate.opsForValue().set(key, EMPTY_VALUE, CACHE_NULL_TTL, TimeUnit.SECONDS);
                return null;
            }
            //数据库存在数据
            this.set(key, list, timeout, unit);

        } catch (InterruptedException e) {
            log.error("query data with mutex list |{}", e.getMessage());
            throw new RuntimeException(e);
        }finally {
            distributedLock.unlock();
        }
        return list;
    }

    private String getLockKey(String key){
        return key.concat(LOCK_SUFFIX);
    }


    /**
     * 将对象类型的json字符串转换成泛型类型
     * @param obj 未知类型对象
     * @param type 泛型Class类型
     * @return 泛型对象
     * @param <R> 泛型
     */
    private <R> R getResult(Object obj, Class<R> type){
        if (obj == null){
            return null;
        }
        //简单类型
        if (TypeConversion.isSimpleType(obj)){
            return Convert.convert(type, obj);
        }
        return JSON.parseObject(JSON.toJSONString(obj),type);
    }

    /**
     * 将对象类型的json字符串转换成泛型类型的List集合
     * @param str json字符串
     * @param type 泛型Class类型
     * @return 泛型List集合
     * @param <R> 泛型
     */
    private <R> List<R> getResultList(String str, Class<R> type){
        if (StrUtil.isEmpty(str)){
            return null;
        }
        return JSON.parseArray(str,type);
    }

    /**
     * 获取简单的key
     * @param key key
     * @return 返回key
     */
    private String getKey(String key){
        return getKey(key, null);
    }

    /**
     * 不确定参数类型的情况下，使用MD5计算参数的拼接到Redis中的唯一Key
     * @param keyPrefix 缓存key的前缀
     * @param id 泛型参数
     * @return 拼接好的缓存key
     * @param <ID> 参数泛型类型
     */
    private <ID> String getKey(String keyPrefix, ID id){
        if (id == null){
            return keyPrefix;
        }
        String key = "";
        //简单数据类型与简单字符串
        if (TypeConversion.isSimpleType(id)){
            key = StrUtil.toString(id);
        }else {
            key = SecureUtil.md5(JSON.toJSONString(id));
        }
        if (StrUtil.isEmpty(key)){
            key = "";
        }
        return keyPrefix.concat(key);
    }

    /**
     * 获取要保存到缓存中的value字符串，可能是简单类型，也可能是对象类型，也可能是集合数组等
     * @param value 要保存的value值
     * @return 处理好的字符串
     */
    private String getValue(Object value){
        return TypeConversion.isSimpleType(value) ? String.valueOf(value) : JSON.toJSONString(value);
    }

    public boolean exists(String key) {
        return BooleanUtil.isTrue(stringRedisTemplate.hasKey(key));
    }
}
