package org.example.common.aspect;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.extra.servlet.ServletUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.example.common.anno.Idempotent;
import org.example.common.constant.Constants;
import org.example.common.domain.R;
import org.example.common.lock.DistributedLock;
import org.example.common.lock.factory.DistributedLockFactory;
import org.example.common.util.WebUtil;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.Objects;

/**
 * @author liutao
 * @since 2023/8/17
 */
@Slf4j
@Aspect
@Component
public class IdempotentAspect {
    @Resource
    private DistributedLockFactory distributedLockFactory;
    private static final ExpressionParser PARSER = new SpelExpressionParser();


    @Pointcut("@annotation(org.example.common.anno.Idempotent)")
    public void idempotent() {}


    @Around("idempotent()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {

        Method method = ((MethodSignature) joinPoint.getSignature()).getMethod();
        Idempotent idempotent = method.getAnnotation(Idempotent.class);
        if (Objects.isNull(idempotent)) {
            return joinPoint.proceed();
        }
        String keySuffix = null;
        if (StringUtils.isNotBlank(idempotent.express())) {
            StandardEvaluationContext evaluationContext = getEvaluationContext(joinPoint);
            keySuffix = getElValue(idempotent.express(),evaluationContext);
        }
        String key = getKey(method, idempotent, keySuffix);
        log.info("aop切面redisKey：{}",key);
        DistributedLock distributedLock = distributedLockFactory.getDistributedLock(key);
        try {
            if (distributedLock.tryLock()) {
                return joinPoint.proceed();
            }
            log.info("redis已存在key，重复提交");
            return R.fail("操作频繁,请稍后再试");
        } finally {
            if (distributedLock.isLocked()) {
                distributedLock.unlock();
            }
        }

    }

    private String getKey(Method method, Idempotent idempotent, String keySuffix) {
        String keyPrefix = idempotent.key();
        if (StringUtils.isBlank(keyPrefix)) {
            keyPrefix = Constants.REPEAT_SUBMIT_KEY;
        }
        if (StringUtils.isNotBlank(keySuffix)) {
            return keyPrefix +":"+ method.getDeclaringClass().getSimpleName() +":"+ method.getName() +":"+ keySuffix;
        }
        HttpServletRequest request = WebUtil.getRequest();
        String queryString = WebUtil.getQueryString(request);
        log.info("url参数：{}",queryString);
        String body = ServletUtil.getBody(request);
        log.info("body参数：{}",body);
        String md5 = SecureUtil.md5(queryString + body);
        return keyPrefix + ":" + md5;

    }


    /**
     * 获取方法上的参数
     */
    private StandardEvaluationContext getEvaluationContext(ProceedingJoinPoint joinPoint) {
        StandardEvaluationContext context = new StandardEvaluationContext(joinPoint.getArgs());
        return setContextVariables(context,joinPoint);
    }

    private StandardEvaluationContext setContextVariables(StandardEvaluationContext standardEvaluationContext,
                                                          JoinPoint joinPoint) {

        Object[] args = joinPoint.getArgs();
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method targetMethod = methodSignature.getMethod();
        LocalVariableTableParameterNameDiscoverer parameterNameDiscoverer = new LocalVariableTableParameterNameDiscoverer();
        String[] parametersName = parameterNameDiscoverer.getParameterNames(targetMethod);

        if (ArrayUtil.isEmpty(args) || ArrayUtil.isEmpty(parametersName)) {
            return standardEvaluationContext;
        }
        for (int i = 0; i < args.length; i++) {
            standardEvaluationContext.setVariable(parametersName[i], args[i]);
        }
        return standardEvaluationContext;
    }

    /**
     * 通过key SEL表达式获取值
     *
     */
    private String getElValue(String key, StandardEvaluationContext context) {
        if (StringUtils.isBlank(key)) {
            return "";
        }
        Expression exp = PARSER.parseExpression(key);
        return exp.getValue(context, String.class);

    }
}
