package org.example.common.thirty.impl;

import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.net.URLEncodeUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.example.common.thirty.TycService;
import org.example.common.thirty.entity.*;
import org.example.common.util.HttpUtil;
import org.example.common.util.JsonUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;

/**
 * @author liutao
 * @since 2024/4/22
 */
@Slf4j
@Service
@ConditionalOnProperty(name = "tyc.enable", havingValue = "true")
public class TycServiceImpl implements TycService {


    private static final String TYC_TOKEN_NAME = "Authorization";
    private static final String REASON = "reason";
    private static final String RESULT = "result";
    private static final String ITEMS = "items";
    private static final Integer MAX_PAGE_SIZE = 20;
    private static final String SUCCESS_REASON = "ok";
    private static final String ERROR_CODE_KEY = "error_code";


    @Value("${tyc.token}")
    private String token;

    @Override
    public List<CompanySearchData> search(String name, Integer pageNum) {
        try {
            String url = "https://open.api.tianyancha.com/services/open/search/2.0?word=%s&pageSize=%s&pageNum=%s";
            String responseBody = HttpUtil.get(String.format(url, name, MAX_PAGE_SIZE, pageNum), MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanySearchResponse<CompanySearchData> response = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanySearchResponse<CompanySearchData>>() {});
            if (response == null) {
                return Collections.emptyList();
            }
            String reason = response.getReason();
            if (SUCCESS_REASON.equals(reason)) {
                TycCompanySearchResponse.TycResult<CompanySearchData> result = response.getResult();
                if (result != null) {
                    return result.getItems();
                }
            }else {
                log.error("天眼查接口异常：{}",reason);
            }
        } catch (Exception e) {
            log.error("天眼查接口报错：{}", e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    @Override
    public CompanyBaseInfoData getCompanyBaseInfo(String keyword) {
        try {
            String url = "https://open.api.tianyancha.com/services/open/ic/baseinfo/normal?keyword="+ URLEncodeUtil.encode(keyword);
            String responseBody = HttpUtil.get(url, MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanyBaseInfoData> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanyBaseInfoData>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业基本信息接口异常：{}", resp.getReason());
        } catch (Exception e) {
            log.error("天眼查查询企业基本信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }

    @Override
    public CompanyProductResult getCompanyProduct(String keyword, Integer pageNum) {
        try {
            String url = "https://open.api.tianyancha.com/services/open/m/appbkInfo/2.0?pageSize=20&keyword=%s&pageNum=%s";
            String responseBody = HttpUtil.get(String.format(url, URLEncodeUtil.encode(keyword), pageNum), MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanyProductResult> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanyProductResult>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业产品信息接口异常：{}", resp.getReason());
        } catch (UtilException e) {
            log.error("天眼查查询企业产品信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }

    @Override
    public CompanyCustomerResult getCompanyCustomer(String keyword, Integer pageNum, Integer year) {
        String url = "https://open.api.tianyancha.com/services/open/m/customer/2.0?year=%s&pageSize=20&keyword=%s&pageNum=%s";
        try {
            String responseBody = HttpUtil.get(String.format(url, year, URLEncodeUtil.encode(keyword), pageNum), MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanyCustomerResult> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanyCustomerResult>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业客户信息接口异常：{}", resp.getReason());
        }catch (Exception e) {
            log.error("天眼查查询企业客户信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }

    @Override
    public CompanySupplyResult getCompanySupply(String keyword, Integer pageNum, Integer year) {
        String url = "https://open.api.tianyancha.com/services/open/m/supply/2.0?year=%s&pageSize=20&keyword=%s&pageNum=%s";
        try {
            String responseBody = HttpUtil.get(String.format(url, year, URLEncodeUtil.encode(keyword), pageNum), MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanySupplyResult> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanySupplyResult>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业供应商信息接口异常：{}", resp.getReason());
        }catch (Exception e) {
            log.error("天眼查查询企业供应商信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }

    @Override
    public CompanyGroupResult getCompanyGroup(String keyword) {
        String url = "https://open.api.tianyancha.com/services/open/group/base?keyword=%s";
        try {
            String responseBody = HttpUtil.get(String.format(url, URLEncodeUtil.encode(keyword)), MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanyGroupResult> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanyGroupResult>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业集团信息接口异常：{}", resp.getReason());
        }catch (Exception e) {
            log.error("天眼查查询企业集团信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }

    @Override
    public CompanyGroupMemberResult getCompanyGroupMember(Integer type, String uuid, Integer pageNum) {
        String url = "https://open.api.tianyancha.com/services/open/group/member?pageSize=20&type=%s&uuid=%s&pageNum=%s";
        try {
            String responseBody = HttpUtil.get(String.format(url, type,uuid, pageNum),MapUtil.of(TYC_TOKEN_NAME, token));
            TycCompanyInfoResponse<CompanyGroupMemberResult> resp = JsonUtils.parseObject(responseBody, new TypeReference<TycCompanyInfoResponse<CompanyGroupMemberResult>>() {
            });
            if (resp == null) {
                return null;
            }
            Integer errorCode = resp.getErrorCode();
            if (errorCode == 0) {
                return resp.getResult();
            }
            log.error("天眼查查询企业集团成员信息接口异常：{}", resp.getReason());
        }catch (Exception e) {
            log.error("天眼查查询企业集团成员信息接口报错：{}", e.getMessage(), e);
        }
        return null;
    }
}
