/*
 * Decompiled with CFR 0.152.
 */
package org.ehcache.impl.persistence;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import org.ehcache.CachePersistenceException;
import org.ehcache.core.spi.service.LocalPersistenceService;
import org.ehcache.impl.config.persistence.DefaultPersistenceConfiguration;
import org.ehcache.impl.persistence.FileUtils;
import org.ehcache.shadow.org.terracotta.utilities.io.Files;
import org.ehcache.spi.service.MaintainableService;
import org.ehcache.spi.service.Service;
import org.ehcache.spi.service.ServiceProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultLocalPersistenceService
implements LocalPersistenceService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultLocalPersistenceService.class);
    private final File rootDirectory;
    private final File lockFile;
    private final File cleanFile;
    private FileLock lock;
    private RandomAccessFile rw;
    private boolean started;
    private boolean clean;

    public DefaultLocalPersistenceService(DefaultPersistenceConfiguration persistenceConfiguration) {
        if (persistenceConfiguration == null) {
            throw new NullPointerException("DefaultPersistenceConfiguration cannot be null");
        }
        this.rootDirectory = persistenceConfiguration.getRootDirectory();
        this.lockFile = new File(this.rootDirectory, ".lock");
        this.cleanFile = new File(this.rootDirectory, ".clean");
    }

    @Override
    public synchronized void start(ServiceProvider<Service> serviceProvider) {
        this.internalStart();
    }

    @Override
    public synchronized void startForMaintenance(ServiceProvider<? super MaintainableService> serviceProvider, MaintainableService.MaintenanceScope maintenanceScope) {
        this.internalStart();
    }

    private void internalStart() {
        if (!this.started) {
            this.clean = false;
            FileUtils.createLocationIfRequiredAndVerify(this.rootDirectory);
            try {
                if (FileUtils.isDirectoryEmpty(this.rootDirectory.toPath())) {
                    this.clean = true;
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            try {
                this.rw = new RandomAccessFile(this.lockFile, "rw");
            }
            catch (FileNotFoundException e) {
                throw new RuntimeException(e);
            }
            try {
                this.lock = this.rw.getChannel().tryLock();
            }
            catch (OverlappingFileLockException e) {
                throw new RuntimeException("Persistence directory already locked by this process: " + this.rootDirectory.getAbsolutePath(), e);
            }
            catch (Exception e) {
                try {
                    this.rw.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw new RuntimeException("Persistence directory couldn't be locked: " + this.rootDirectory.getAbsolutePath(), e);
            }
            if (this.lock == null) {
                throw new RuntimeException("Persistence directory already locked by another process: " + this.rootDirectory.getAbsolutePath());
            }
            if (this.cleanFile.exists()) {
                try {
                    LOGGER.debug("Clean file exists, trying to delete the file.");
                    Files.delete(this.cleanFile.toPath());
                    this.clean = true;
                    LOGGER.debug("Clean file is deleted.");
                }
                catch (IOException e) {
                    LOGGER.debug("Clean file is not deleted {}.", (Object)this.cleanFile.getPath());
                    throw new RuntimeException(e);
                }
            }
            this.started = true;
            LOGGER.debug("RootDirectory Locked");
        }
    }

    @Override
    public synchronized void stop() {
        if (this.started) {
            try {
                if (this.cleanFile.createNewFile()) {
                    LOGGER.debug("Clean file is created.");
                } else {
                    LOGGER.warn("Clean file still exists on shutdown. This may be due to a filesystem consistency issue, or concurrent modification of the filesystem by another thread and/or process. Be aware for the potential of undetected storage corruption causing cache failures and/or data consistency issues.");
                }
            }
            catch (IOException e) {
                LOGGER.warn("Clean file could not be created on shutdown. This will cause the persisted state to be considered unclean on the next service start, which may result in unplanned data loss. Reason: " + e.toString());
            }
            try {
                this.lock.release();
                this.rw.close();
                try {
                    Files.delete(this.lockFile.toPath());
                }
                catch (IOException e) {
                    LOGGER.debug("Lock file was not deleted {}.", (Object)this.lockFile.getPath());
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Couldn't unlock rootDir: " + this.rootDirectory.getAbsolutePath(), e);
            }
            this.started = false;
            LOGGER.debug("RootDirectory Unlocked");
        }
    }

    File getLockFile() {
        return this.lockFile;
    }

    @Override
    public LocalPersistenceService.SafeSpaceIdentifier createSafeSpaceIdentifier(String owner, String identifier) {
        FileUtils.validateName(owner);
        SafeSpace ss = this.createSafeSpaceLogical(owner, identifier);
        for (File parent = ss.directory.getParentFile(); parent != null; parent = parent.getParentFile()) {
            if (!this.rootDirectory.equals(parent)) continue;
            return new DefaultSafeSpaceIdentifier(ss);
        }
        throw new IllegalArgumentException("Attempted to access file outside the persistence path");
    }

    @Override
    public void createSafeSpace(LocalPersistenceService.SafeSpaceIdentifier safeSpaceId) throws CachePersistenceException {
        if (safeSpaceId == null || !(safeSpaceId instanceof DefaultSafeSpaceIdentifier)) {
            throw new AssertionError((Object)"Invalid safe space identifier. Identifier not created");
        }
        SafeSpace ss = ((DefaultSafeSpaceIdentifier)safeSpaceId).safeSpace;
        FileUtils.create(ss.directory.getParentFile());
        FileUtils.create(ss.directory);
    }

    @Override
    public void destroySafeSpace(LocalPersistenceService.SafeSpaceIdentifier safeSpaceId, boolean verbose) {
        if (safeSpaceId == null || !(safeSpaceId instanceof DefaultSafeSpaceIdentifier)) {
            throw new AssertionError((Object)"Invalid safe space identifier. Identifier not created");
        }
        SafeSpace ss = ((DefaultSafeSpaceIdentifier)safeSpaceId).safeSpace;
        this.destroy(ss, verbose);
    }

    @Override
    public void destroyAll(String owner) {
        File ownerDirectory = new File(this.rootDirectory, owner);
        if (ownerDirectory.exists() && ownerDirectory.isDirectory()) {
            if (FileUtils.tryRecursiveDelete(ownerDirectory)) {
                LOGGER.debug("Destroyed all file based persistence contexts owned by {}", (Object)owner);
            } else {
                LOGGER.warn("Could not delete all file based persistence contexts owned by {}", (Object)owner);
            }
        } else {
            LOGGER.warn("Could not delete all file based persistence contexts owned by {} - is not a directory!", (Object)owner);
        }
    }

    @Override
    public final synchronized boolean isClean() {
        if (this.started) {
            return this.clean;
        }
        throw new IllegalStateException("Service is not running");
    }

    private void destroy(SafeSpace ss, boolean verbose) {
        if (verbose) {
            LOGGER.debug("Destroying file based persistence context for {}", (Object)ss.identifier);
        }
        if (ss.directory.exists() && !FileUtils.tryRecursiveDelete(ss.directory) && verbose) {
            LOGGER.warn("Could not delete directory for context {}", (Object)ss.identifier);
        }
    }

    private SafeSpace createSafeSpaceLogical(String owner, String identifier) {
        File ownerDirectory = new File(this.rootDirectory, owner);
        File directory = new File(ownerDirectory, FileUtils.safeIdentifier(identifier));
        return new SafeSpace(identifier, directory);
    }

    private static final class DefaultSafeSpaceIdentifier
    implements LocalPersistenceService.SafeSpaceIdentifier {
        private final SafeSpace safeSpace;

        private DefaultSafeSpaceIdentifier(SafeSpace safeSpace) {
            this.safeSpace = safeSpace;
        }

        public String toString() {
            return this.safeSpace.identifier;
        }

        @Override
        public File getRoot() {
            return this.safeSpace.directory;
        }
    }

    private static final class SafeSpace {
        private final String identifier;
        private final File directory;

        private SafeSpace(String identifier, File directory) {
            this.directory = directory;
            this.identifier = identifier;
        }
    }
}

