/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.implementation.LoggingUtil;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.ByteArrayOutputStream;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintJson;

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        this(httpLogOptions, false);
    }

    private HttpLoggingPolicy(HttpLogOptions httpLogOptions, boolean prettyPrintJson) {
        this.prettyPrintJson = prettyPrintJson;
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames().stream().map(headerName -> headerName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames().stream().map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
        }
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        ClientLogger logger = new ClientLogger((String)context.getData("caller-method").orElse(""));
        long startNs = System.nanoTime();
        return this.logRequest(logger, context.getHttpRequest()).then(next.process()).flatMap(response -> this.logResponse(logger, (HttpResponse)response, startNs)).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private Mono<Void> logRequest(ClientLogger logger, HttpRequest request) {
        long contentLength;
        int numericLogLevel = LoggingUtil.getEnvironmentLoggingLevel().getLogLevel();
        if (this.shouldLoggingBeSkipped(numericLogLevel)) {
            return Mono.empty();
        }
        StringBuilder requestLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            requestLogMessage.append("--> ").append((Object)request.getHttpMethod()).append(" ").append(this.getRedactedUrl(request.getUrl())).append(System.lineSeparator());
        }
        this.addHeadersToLogMessage(request.getHeaders(), requestLogMessage, numericLogLevel);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            return this.logAndReturn(logger, requestLogMessage, null);
        }
        if (request.getBody() == null) {
            requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            return this.logAndReturn(logger, requestLogMessage, null);
        }
        String contentType = request.getHeaders().getValue("Content-Type");
        if (this.shouldBodyBeLogged(contentType, contentLength = this.getContentLength(logger, request.getHeaders()))) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            request.setBody((Flux<ByteBuffer>)request.getBody().doOnNext(byteBuffer -> {
                for (int i = byteBuffer.position(); i < byteBuffer.limit(); ++i) {
                    outputStream.write(byteBuffer.get(i));
                }
            }).doFinally(ignored -> {
                requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentType, new String(outputStream.toByteArray(), StandardCharsets.UTF_8))).append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                logger.info(requestLogMessage.toString(), new Object[0]);
            }));
            return Mono.empty();
        }
        requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
        return this.logAndReturn(logger, requestLogMessage, null);
    }

    private Mono<HttpResponse> logResponse(ClientLogger logger, HttpResponse response, long startNs) {
        long contentLength;
        int numericLogLevel = LoggingUtil.getEnvironmentLoggingLevel().getLogLevel();
        if (this.shouldLoggingBeSkipped(numericLogLevel)) {
            return Mono.just((Object)response);
        }
        long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
        String contentLengthString = response.getHeaderValue("Content-Length");
        String bodySize = CoreUtils.isNullOrEmpty(contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
        StringBuilder responseLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(this.getRedactedUrl(response.getRequest().getUrl())).append(" (").append(tookMs).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
        }
        this.addHeadersToLogMessage(response.getHeaders(), responseLogMessage, numericLogLevel);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            responseLogMessage.append("<-- END HTTP");
            return this.logAndReturn(logger, responseLogMessage, response);
        }
        String contentTypeHeader = response.getHeaderValue("Content-Type");
        if (this.shouldBodyBeLogged(contentTypeHeader, contentLength = this.getContentLength(logger, response.getHeaders()))) {
            HttpResponse bufferedResponse = response.buffer();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            return bufferedResponse.getBody().doOnNext(byteBuffer -> {
                for (int i = byteBuffer.position(); i < byteBuffer.limit(); ++i) {
                    outputStream.write(byteBuffer.get(i));
                }
            }).doFinally(ignored -> {
                responseLogMessage.append("Response body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentTypeHeader, new String(outputStream.toByteArray(), StandardCharsets.UTF_8))).append(System.lineSeparator()).append("<-- END HTTP");
                logger.info(responseLogMessage.toString(), new Object[0]);
            }).then(Mono.just((Object)bufferedResponse));
        }
        responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
        return this.logAndReturn(logger, responseLogMessage, response);
    }

    private <T> Mono<T> logAndReturn(ClientLogger logger, StringBuilder logMessageBuilder, T data) {
        logger.info(logMessageBuilder.toString(), new Object[0]);
        return Mono.justOrEmpty(data);
    }

    private boolean shouldLoggingBeSkipped(int environmentLogLevel) {
        return environmentLogLevel > LogLevel.INFORMATIONAL.getLogLevel();
    }

    private String getRedactedUrl(URL url) {
        return UrlBuilder.parse(url).setQuery(this.getAllowedQueryString(url.getQuery())).toString();
    }

    private String getAllowedQueryString(String queryString) {
        String[] queryParams;
        if (CoreUtils.isNullOrEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (this.allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private void addHeadersToLogMessage(HttpHeaders headers, StringBuilder sb, int logLevel) {
        if (!this.httpLogDetailLevel.shouldLogHeaders() || logLevel > LogLevel.VERBOSE.getLogLevel()) {
            return;
        }
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (this.allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private String prettyPrintIfNeeded(ClientLogger logger, String contentType, String body) {
        String result = body;
        if (this.prettyPrintJson && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (CoreUtils.isNullOrEmpty(contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }

    private boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }
}

