package com.ites.mail.sender;

import cn.hutool.core.util.BooleanUtil;
import com.ites.mail.exception.MailSenderException;
import com.ites.mail.request.MailSenderRequest;
import com.ites.mail.util.EmailValidateUtil;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author liutao
 * @since 2023/12/29
 */
public abstract class AbstractMailSenderService implements MailSenderService{


    protected void check(MailSenderRequest request) {
        check(request.getTo(),request.getSubject(),request.getText());
    }


    protected void check(List<String> to, String subject, String text) {
        if (CollectionUtils.isEmpty(to)) {
            throw new MailSenderException("邮件接收人不能为空");
        }
        List<String> errorEmails = to.stream().filter(EmailValidateUtil::isNotEmail).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(errorEmails)) {
            throw new MailSenderException("邮件接收人格式不正确：" + String.join(",", errorEmails));
        }
        if (!StringUtils.hasText(subject)) {
            throw new MailSenderException("邮件主题不能为空");
        }
        if (!StringUtils.hasText(text)) {
            throw new MailSenderException("邮件内容不能为空");
        }
    }

    protected void addHeaders(MailSenderRequest request, MimeMessage message) throws Exception {
        if (BooleanUtil.isTrue(request.getIsMarket())) {
            // 营销类消息，例如：广告，期刊等
            message.addHeader("X-Qos", "bulk");
        }
        //联系人列表名称，如果不存在，会自动创建
        message.addHeader("X-List", MimeUtility.encodeText(request.getTaskName() == null ? "singleSend" : request.getTaskName()));
        //邮件名称，如果不存在，会自动创建
        message.addHeader("X-Campaign", MimeUtility.encodeText(request.getSubject()));
        //自定义文本标识
        message.addHeader("X-Tracking-ID", MimeUtility.encodeText(request.getBusinessId() == null ? "0" : String.valueOf(request.getBusinessId())));
        //自定义文本数据
        message.addHeader("X-Tracking-Data", MimeUtility.encodeText(request.getBusinessId() == null ? "0" : String.valueOf(request.getBusinessId())));
    }
}
