package com.ites.mail.config;

import com.ites.mail.enums.MailTypeEnum;
import com.ites.mail.sender.MailSenderService;
import com.ites.mail.sender.impl.SpringMailSenderServiceImpl;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.JavaMailSenderImpl;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * @author liutao
 * @since 2025/1/8
 */
@Configuration
@EnableConfigurationProperties(MailProperties.class)
public class MailConfiguration {



    @Bean
    public Map<Integer, JavaMailSender> javaMailSenderMap(MailProperties mailProperties) {

        Map<Integer, JavaMailSender> mailSenderMap = new HashMap<>();

        JavaMailSenderImpl notifySender = new JavaMailSenderImpl();
        applyProperties(mailProperties.getNotify(),notifySender);

        mailSenderMap.put(MailTypeEnum.NOTIFY.getType(), notifySender);

        JavaMailSenderImpl marketSender = new JavaMailSenderImpl();
        applyProperties(mailProperties.getMarket(),marketSender);

        mailSenderMap.put(MailTypeEnum.MARKET.getType(), marketSender);
        return mailSenderMap;
    }

    @Bean
    @ConditionalOnMissingBean
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        Jackson2JsonRedisSerializer<Object> jackson2JsonRedisSerializer = new Jackson2JsonRedisSerializer<>(Object.class);
        redisTemplate.setValueSerializer(jackson2JsonRedisSerializer);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashValueSerializer(jackson2JsonRedisSerializer);
        redisTemplate.afterPropertiesSet();
        return redisTemplate;
    }

    @Bean
    public MailSenderService mailSenderService(Map<Integer, JavaMailSender> javaMailSenderMap,
                                               MailProperties mailProperties,
                                               StringRedisTemplate redisTemplate) {
        return new SpringMailSenderServiceImpl(javaMailSenderMap,mailProperties,redisTemplate);
    }


    private void applyProperties(MailProperties.Account properties, JavaMailSenderImpl sender) {
        sender.setHost(properties.getHost());
        if (properties.getPort() != null) {
            sender.setPort(properties.getPort());
        }
        sender.setUsername(properties.getUsername());
        sender.setPassword(properties.getPassword());
        sender.setProtocol(properties.getProtocol());
        if (properties.getDefaultEncoding() != null) {
            sender.setDefaultEncoding(properties.getDefaultEncoding().name());
        }
        if (!properties.getProperties().isEmpty()) {
            sender.setJavaMailProperties(asProperties(properties.getProperties()));
        }
    }


    private Properties asProperties(Map<String, String> source) {
        Properties properties = new Properties();
        properties.putAll(source);
        return properties;
    }

}
