package com.ites.mail.sender.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.StrUtil;
import com.ites.mail.attachment.Attachment;
import com.ites.mail.attachment.FileAttachment;
import com.ites.mail.attachment.UrlAttachment;
import com.ites.mail.config.MailProperties;
import com.ites.mail.constant.EmailConstant;
import com.ites.mail.enums.MailTypeEnum;
import com.ites.mail.exception.MailSenderException;
import com.ites.mail.request.MailSenderRequest;
import com.ites.mail.sender.AbstractMailSenderService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.FileUrlResource;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scripting.support.ResourceScriptSource;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author liutao
 * @since 2023/8/30
 */
@Slf4j
public class SpringMailSenderServiceImpl extends AbstractMailSenderService {

    private final Map<Integer, JavaMailSender> javaMailSenderMap;
    private final MailProperties mailProperties;
    private final StringRedisTemplate redisTemplate;

    private final DefaultRedisScript<String> checkEmailSendCountScript;
    private final DefaultRedisScript<Long> batchAddEmailSendCountScript;


    public SpringMailSenderServiceImpl(Map<Integer, JavaMailSender> javaMailSenderMap, MailProperties mailProperties,
                                       StringRedisTemplate redisTemplate) {
        this.javaMailSenderMap = javaMailSenderMap;
        this.mailProperties = mailProperties;
        this.redisTemplate = redisTemplate;

        batchAddEmailSendCountScript = new DefaultRedisScript<>();
        batchAddEmailSendCountScript.setScriptSource(new ResourceScriptSource(new ClassPathResource("lua/batch_add_email_send_count.lua")));
        batchAddEmailSendCountScript.setResultType(Long.class);

        checkEmailSendCountScript = new DefaultRedisScript<>();
        checkEmailSendCountScript.setScriptSource(new ResourceScriptSource(new ClassPathResource("lua/check_email_send_count.lua")));
        checkEmailSendCountScript.setResultType(String.class);
    }

    @Override
    public boolean send(MailSenderRequest request) {
        // 检查参数
        check(request);
        // 排除超过发送次数的邮箱
        List<String> excludes = filterOverSendCountEmail(request);
        log.info("达到发送次数邮箱：{}",excludes);
        if (CollectionUtils.isEmpty(request.getTo())) {
            return true;
        }
        int mailType = request.getIsMarket() ? MailTypeEnum.MARKET.getType() : MailTypeEnum.NOTIFY.getType();
        JavaMailSender javaMailSender = javaMailSenderMap.get(mailType);
        // 获取MimeMessage对象
        MimeMessage message = javaMailSender.createMimeMessage();
        MimeMessageHelper messageHelper;
        try {
            // 添加头
            addHeaders(request,message);
            // 创建MimeMessageHelper对象
            messageHelper = new MimeMessageHelper(message, true);
            // 邮件发送人
            settingFrom(messageHelper,request);
            // 邮件接收人,设置多个收件人地址
            messageHelper.setTo(buildAddress(request.getTo()));
            // 邮件抄送人
            messageHelper.setCc(buildAddress(request.getCc()));
            // 设置附件
            addAttachment(messageHelper,request.getAttachmentList());
            // 邮件主题
            message.setSubject(request.getSubject());
            // 邮件内容，html格式
            messageHelper.setText(request.getText(), request.getIsHtml());
            // 发送
            javaMailSender.send(message);
            // 更新邮箱发送次数
            if (request.getIsMarket()) {
                batchAddMobileSendCount(request.getTo());
            }
            // 日志信息
            log.info("邮件已经发送。");
            return true;
        } catch (Exception e) {
            log.error("发送邮件时发生异常！", e);
            return false;
        }
    }

    private List<String> filterOverSendCountEmail(MailSenderRequest request) {

        if (!request.getIsMarket()) {
            return Collections.emptyList();
        }
        List<String> to = request.getTo();
        String mobilesStr = String.join(",", to);
        long expireSeconds = LocalDateTimeUtil.between(LocalDateTime.now(), LocalDateTime.now().withHour(23).withMinute(59).withSecond(59), ChronoUnit.SECONDS);

        // 执行Lua脚本
        String result = redisTemplate.execute(checkEmailSendCountScript,
                Collections.singletonList(EmailConstant.EMAIL_CACHE_SEND_COUNT_KEY),
                String.valueOf(EmailConstant.MARKET_MAX_SEND_COUNT),
                mobilesStr,
                String.valueOf(expireSeconds));

        if (result != null && !result.isEmpty()) {
            List<String> excludes = Arrays.stream(result.split(",")).collect(Collectors.toList());
            to.removeIf(excludes::contains);
            request.setTo(to);
            return excludes;
        } else {
            return Collections.emptyList();
        }
    }

    private void batchAddMobileSendCount(List<String> mobiles) {
        String mobilesStr = String.join(",", mobiles);
        // 执行Lua脚本
        redisTemplate.execute(batchAddEmailSendCountScript,
                Collections.singletonList(EmailConstant.EMAIL_CACHE_SEND_COUNT_KEY),
                mobilesStr);
    }


    @Override
    public boolean sendHtmlEmail(String email, String subject, String content) {
        ArrayList<String> to = new ArrayList<>();
        to.add(email);
        MailSenderRequest request = MailSenderRequest.builder()
                .subject(subject)
                .text(content)
                .isMarket(false)
                .to(to)
                .isHtml(true).build();
        return send(request);
    }

    private void settingFrom(MimeMessageHelper messageHelper, MailSenderRequest request) throws Exception {
        MailProperties.Account account = request.getIsMarket() ? mailProperties.getMarket() : mailProperties.getNotify();
        if (StrUtil.isBlank(account.getFrom())) {
            throw new MailSenderException("邮件发送人不能为空");
        }
        String fromEmail = account.getFrom();
        String fromPersonal = account.getFromPersonal();
        if (StringUtils.hasText(fromPersonal)) {
            messageHelper.setFrom(fromEmail,fromPersonal);
        }
    }

    private void addAttachment(MimeMessageHelper messageHelper, List<Attachment> attachmentList) throws MalformedURLException, MessagingException, UnsupportedEncodingException {
        if (CollectionUtils.isEmpty(attachmentList)) {
            return;
        }
        for (Attachment attachment : attachmentList) {
            String name = attachment.getName();
            Object data = attachment.getData();
            Assert.isTrue(StringUtils.hasText(name),"附件名称不能为空");
            Assert.isTrue(Objects.nonNull(data),"附件不能为空");
            if (attachment instanceof FileAttachment) {
                messageHelper.addAttachment(MimeUtility.encodeText(name), new FileSystemResource((File) data), attachment.getContentType());
            }
            if (attachment instanceof UrlAttachment) {
                messageHelper.addAttachment(MimeUtility.encodeText(name), new FileUrlResource(new URL(data.toString())), attachment.getContentType());
            }
        }
    }

    private InternetAddress[] buildAddress(List<String> toList) {
        if (CollectionUtils.isEmpty(toList)) {
            return new InternetAddress[0];
        }
        List<InternetAddress> list = new ArrayList<>();
        try {
            for (String to : toList) {
                list.add(new InternetAddress(to));
            }
        } catch (AddressException e) {
            throw new RuntimeException(e);
        }
        return list.toArray(new InternetAddress[0]);
    }
}
