package com.simm.common.utils;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.httpclient.ConnectTimeoutException;
import org.apache.commons.httpclient.DefaultHttpMethodRetryHandler;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpClientError;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.params.HttpConnectionParams;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.httpclient.protocol.ControllerThreadSocketFactory;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.SecureProtocolSocketFactory;
import org.apache.commons.lang3.StringUtils;

/**
 * 
 * @Description Http
 *
 * @author YingJH
 * @Date 2017年9月21日
 * @version v1.0
 */
public class HttpUtil {

	public static enum HttpMethodEnum {
		POST, GET
	}

	public static enum HttpContentTypeEnum {
		JSON("application/json"), TEXT("");

		private final String value;

		HttpContentTypeEnum(String value) {
			this.value = value;
		}

		public String getValue() {
			return value;
		}
	}

	public static enum CharSetEnum {
		UTF8("UTF-8"), GBK("GBK"), GB2312("GB2312");

		private final String value;

		CharSetEnum(String value) {
			this.value = value;
		}

		public String getValue() {
			return value;
		}
	}

	/**
	 * 发送http请求 GET
	 * 
	 * @param urlString
	 * @param param
	 * @param hmEnum
	 * @return
	 */
	public static String HttpConnect(String urlString) {
		return HttpConnect(urlString, null, HttpMethodEnum.GET, HttpContentTypeEnum.JSON, CharSetEnum.UTF8);
	}

	/**
	 * 发送http请求
	 * 
	 * @param urlString
	 * @param param
	 * @param hmEnum
	 * @return
	 */
	public static String HttpConnect(String urlString, String param, HttpMethodEnum hmEnum) {
		return HttpConnect(urlString, param, hmEnum, HttpContentTypeEnum.JSON, CharSetEnum.UTF8);
	}

	/**
	 * 发送http请求 带ContentType
	 * 
	 * @param urlString
	 * @param param
	 * @param hmEnum
	 * @return
	 */
	public static String HttpConnect(String urlString, String param, HttpMethodEnum hmEnum, HttpContentTypeEnum hctEnum) {
		return HttpConnect(urlString, param, hmEnum, HttpContentTypeEnum.JSON, CharSetEnum.UTF8);
	}

	/**
	 * 发送http请求 带CharSet
	 * 
	 * @param urlString
	 * @param param
	 * @param hmEnum
	 * @return
	 */
	public static String HttpConnect(String urlString, String param, HttpMethodEnum hmEnum, CharSetEnum csEnum) {
		return HttpConnect(urlString, param, hmEnum, HttpContentTypeEnum.JSON, CharSetEnum.UTF8);
	}

	/**
	 * 发送http请求 带ContentType 、带CharSet
	 * 
	 * @param urlString
	 * @param param
	 * @param hmEnum
	 * @param hctEnum
	 * @param csEnum
	 * @return
	 */
	public static String HttpConnect(String urlString, String param, HttpMethodEnum hmEnum, HttpContentTypeEnum hctEnum, CharSetEnum csEnum) {
		HttpClient httpclient;
		HttpMethod httpMethod;
		ProtocolSocketFactory fcty;
		StringRequestEntity requestEntity;
		int statusCode;

		httpMethod = null;
		try {

			if (urlString.toLowerCase().indexOf("https") != -1) {
				// download token
				fcty = new MySecureProtocolSocketFactory();
				Protocol.registerProtocol("https", new Protocol("https", fcty, 443));
			}

			// send
			httpclient = new HttpClient();
			if (hmEnum.equals(HttpMethodEnum.GET)) { // GET
				httpMethod = new GetMethod(urlString);
				httpMethod.getParams().setParameter(HttpMethodParams.RETRY_HANDLER, new DefaultHttpMethodRetryHandler(3, false));
			} else if (hmEnum.equals(HttpMethodEnum.POST)) { // POST
				httpMethod = new PostMethod(urlString);
				requestEntity = new StringRequestEntity(param, hctEnum.getValue(), csEnum.getValue());
				((PostMethod) httpMethod).setRequestEntity(requestEntity);
			}

			// callback
			if (null != httpMethod) {
				statusCode = httpclient.executeMethod(httpMethod);
				if (statusCode == HttpStatus.SC_OK) {
					return new String(httpMethod.getResponseBody(), "UTF-8");
				} else {
					throw new Exception(statusCode + "---->" + httpMethod.getStatusLine());
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (null != httpMethod) {
				httpMethod.releaseConnection();
			}
		}

		return null;
	}

	/**
	 * 获取用户IP地址
	 * 
	 * @return
	 */
	public static String getClientIp(HttpServletRequest request) {
		String ip = request.getHeader("x-forwarded-for");
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("Proxy-Client-IP");
		}

		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("WL-Proxy-Client-IP");
		}

		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getRemoteAddr();
		}

		if (StringUtils.isEmpty(ip)) {
			ip = ip.split(",")[0];
		}

		return ip;
	}

	/**
	 * 判断ajax请求
	 * 
	 * @param request
	 * @return
	 */
	public static boolean isAjax(HttpServletRequest request) {
		return (request.getHeader("X-Requested-With") != null && "XMLHttpRequest".equals(request.getHeader("X-Requested-With").toString()));
	}

}

class MyX509TrustManager implements X509TrustManager {
	public void checkClientTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
	}

	public void checkServerTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
	}

	public X509Certificate[] getAcceptedIssuers() {
		return null;
	}
}

class MySecureProtocolSocketFactory implements SecureProtocolSocketFactory {
	private SSLContext sslContext = null;

	public MySecureProtocolSocketFactory() {
	}

	private static SSLContext createEasySSLContext() {
		try {
			SSLContext context = SSLContext.getInstance("SSL");
			context.init(null, new TrustManager[] { new MyX509TrustManager() }, null);
			return context;
		} catch (Exception e) {
			throw new HttpClientError(e.toString());
		}
	}

	private SSLContext getSSLContext() {
		if (this.sslContext == null) {
			this.sslContext = createEasySSLContext();
		}
		return this.sslContext;
	}

	public Socket createSocket(String host, int port, InetAddress clientHost, int clientPort) throws IOException, UnknownHostException {
		return getSSLContext().getSocketFactory().createSocket(host, port, clientHost, clientPort);
	}

	public Socket createSocket(final String host, final int port, final InetAddress localAddress, final int localPort, final HttpConnectionParams params) throws IOException, UnknownHostException,
			ConnectTimeoutException {
		if (params == null) {
			throw new IllegalArgumentException("Parameters may not be null");
		}
		int timeout = params.getConnectionTimeout();
		if (timeout == 0) {
			return createSocket(host, port, localAddress, localPort);
		} else {
			return ControllerThreadSocketFactory.createSocket(this, host, port, localAddress, localPort, timeout);
		}
	}

	public Socket createSocket(String host, int port) throws IOException, UnknownHostException {
		return getSSLContext().getSocketFactory().createSocket(host, port);
	}

	public Socket createSocket(Socket socket, String host, int port, boolean autoClose) throws IOException, UnknownHostException {
		return getSSLContext().getSocketFactory().createSocket(socket, host, port, autoClose);
	}
}