package com.simm.common.utils;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.List;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.BucketInfo;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;

/**
 * 
 * @Description 阿里 OSS
 *
 * @author YingJH
 * @Date 2017年12月12日
 * @version v1.0
 */
public class OssUtil {
	//文件分类
    public static final String ALIYUN_OSS_PATH_TEMPLATE = "hive/template/";
    
	// endpoint是访问OSS的域名。如果您已经在OSS的控制台上 创建了Bucket，请在控制台上查看域名。
	// 如果您还没有创建Bucket，endpoint选择请参看文档中心的“开发人员指南 > 基本概念 > 访问域名”，
	// 链接地址是：https://help.aliyun.com/document_detail/oss/user_guide/oss_concept/endpoint.html?spm=5176.docoss/user_guide/endpoint_region
	// endpoint的格式形如“http://oss-cn-hangzhou.aliyuncs.com/”，注意http://后不带bucket名称，
	// 比如“http://bucket-name.oss-cn-hangzhou.aliyuncs.com”，是错误的endpoint，请去掉其中的“bucket-name”。
	private static String endpoint = PropertiesUtil.getKey("endpoint");
	private final static String url = PropertiesUtil.getKey("ossUrl");

	// accessKeyId和accessKeySecret是OSS的访问密钥，您可以在控制台上创建和查看，
	// 创建和查看访问密钥的链接地址是：https://ak-console.aliyun.com/#/。
	// 注意：accessKeyId和accessKeySecret前后都没有空格，从控制台复制时请检查并去除多余的空格。
	private static String accessKeyId = PropertiesUtil.getKey("aliAppId");
	private static String accessKeySecret = PropertiesUtil.getKey("aliAppSecret");

	private static String bucketName = PropertiesUtil.getKey("bucketName");

	// Bucket用来管理所存储Object的存储空间，详细描述请参看“开发人员指南 > 基本概念 > OSS基本概念介绍”。
	// Bucket命名规范如下：只能包括小写字母，数字和短横线（-），必须以小写字母或者数字开头，长度必须在3-63字节之间。

	// Object是OSS存储数据的基本单元，称为OSS的对象，也被称为OSS的文件。详细描述请参看“开发人员指南 > 基本概念 >
	// OSS基本概念介绍”。
	// Object命名规范如下：使用UTF-8编码，长度必须在1-1023字节之间，不能以“/”或者“\”字符开头。

//	private static OSSClient ossClient;

	public static OSSClient getOSSClient() {
		OSSClient ossClient = new OSSClient(endpoint, accessKeyId, accessKeySecret);
		return ossClient;
	}

	public static void shutdown(OSSClient ossClient) {
		ossClient.shutdown();
	}

	public static boolean existAndCreateBucket(String bucketName) {
		OSSClient ossClient= getOSSClient();
		// 判断Bucket是否存在。详细请参看“SDK手册 > Java-SDK > 管理Bucket”。
		// 链接地址是：https://help.aliyun.com/document_detail/oss/sdk/java-sdk/manage_bucket.html?spm=5176.docoss/sdk/java-sdk/init
		if (ossClient.doesBucketExist(bucketName)) {
			return true;
		}
		// 创建Bucket。详细请参看“SDK手册 > Java-SDK > 管理Bucket”。
		// 链接地址是：https://help.aliyun.com/document_detail/oss/sdk/java-sdk/manage_bucket.html?spm=5176.docoss/sdk/java-sdk/init
		ossClient.createBucket(bucketName);
		shutdown(ossClient);
		return false;
	}

	public static BucketInfo getBucketInfo(String bucketName) {
		OSSClient ossClient= getOSSClient();
		// 链接地址是：https://help.aliyun.com/document_detail/oss/sdk/java-sdk/manage_bucket.html?spm=5176.docoss/sdk/java-sdk/init
		BucketInfo info = ossClient.getBucketInfo(bucketName);
		// System.out.println("Bucket " + bucketName + "的信息如下：");
		// System.out.println("\t数据中心：" + info.getBucket().getLocation());
		// System.out.println("\t创建时间：" + info.getBucket().getCreationDate());
		// System.out.println("\t用户标志：" + info.getBucket().getOwner());
		shutdown(ossClient);
		return info;
	}

	public static List<OSSObjectSummary> getBucketObjectInfo(String bucketName) {
		OSSClient ossClient= getOSSClient();
		// 查看Bucket中的Object。详细请参看“SDK手册 > Java-SDK > 管理文件”。
		// 链接地址是：https://help.aliyun.com/document_detail/oss/sdk/java-sdk/manage_object.html?spm=5176.docoss/sdk/java-sdk/manage_bucket
		ObjectListing objectListing = ossClient.listObjects(bucketName);
		List<OSSObjectSummary> objectSummary = objectListing.getObjectSummaries();
		// System.out.println("您有以下Object：");
		// for (OSSObjectSummary object : objectSummary) {
		// System.out.println("\t" + object.getKey());
		// }
		shutdown(ossClient);
		return objectSummary;
	}

	public static String uploadObject(InputStream is, String filePath,String bucketNameCite) throws UnsupportedEncodingException {
		OSSClient ossClient= getOSSClient();
		String fileUrl = null;
		if(StringUtil.isBlank(bucketNameCite)){
			ossClient.putObject(bucketName, filePath, is);
			fileUrl ="http://"+bucketName+"."+url+filePath;
		}else{
			ossClient.putObject(bucketNameCite, filePath, is);
			fileUrl ="http://"+bucketNameCite+"."+url+filePath;
		}
		if(is != null){
			try {
				is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		shutdown(ossClient);
		return fileUrl;
	}

	public static String uploadObject(String serverFilePath, String filePath,String bucketNameCite) {
		OSSClient ossClient= getOSSClient();
		String fileUrl = null;
		if(StringUtil.isBlank(bucketNameCite)){
			ossClient.putObject(bucketName, filePath, new File(serverFilePath));
			fileUrl = "http://"+bucketName+"."+url+filePath;
		}else {
			ossClient.putObject(bucketNameCite, filePath, new File(serverFilePath));
			fileUrl = "http://"+bucketNameCite+"."+url+filePath;
		}
		
		shutdown(ossClient);
		return fileUrl;
	}

	public static StringBuilder downloadObject(String fileName,String bucketNameCite) throws IOException {
		OSSClient ossClient= getOSSClient();
		OSSObject ossObject = null;
		if(StringUtil.isBlank(bucketNameCite)){
			ossObject = ossClient.getObject(bucketName, fileName);
		}else {
			ossObject = ossClient.getObject(bucketNameCite, fileName);
		}
		InputStream inputStream = ossObject.getObjectContent();
		StringBuilder objectContent = new StringBuilder();
		BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
		while (true) {
			String line = reader.readLine();
			if (line == null)
				break;
			objectContent.append(line);
		}
		reader.close();
		inputStream.close();
		shutdown(ossClient);
		return objectContent;

	}

	public static void deleteObject(String fileName,String bucketNameCite) {
		OSSClient ossClient= getOSSClient();
		if(StringUtil.isNotBlank(bucketNameCite)){
			ossClient.deleteObject(bucketNameCite, fileName);
		}
		shutdown(ossClient);
	} 
}
