package org.example.common.thirty.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Dict;
import org.example.common.enums.ShortLinkChannelEnum;
import org.example.common.exception.ApiCallException;
import org.example.common.exception.ServiceException;
import org.example.common.exception.ShortUrlException;
import org.example.common.thirty.ShortLinkService;
import org.example.common.thirty.entity.BaiduShortLinkResponse;
import org.example.common.thirty.entity.ShortLinkGenerateRequest;
import org.example.common.util.HttpUtil;
import org.example.common.util.JsonUtils;
import org.example.common.util.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author liutao
 * @since 2024/3/8
 */
@Service
public class BaiduShortLinkServiceImpl implements ShortLinkService {

    private static final String HOST = "https://dwz.cn/api/v3/short-urls";
    private static final String TOKEN = "95a4fd37cd17d26f8467b9d30d58ecd9";
    private static final String CONTENT_TYPE = "application/json; charset=UTF-8";
    private static final String TERM_OF_VALIDITY = "1-year";
    private static final String SHORT_URL_KEY = "ShortUrl";
    private static final Map<String, String> HEADERS = new HashMap<>(3);

    @PostConstruct
    public void initHeaders() {
        HEADERS.put("Dwz-Token",TOKEN);
        HEADERS.put("Content-Language","zh");
    }

    @Override
    public String create(ShortLinkGenerateRequest request) {
        return create(request.getLongUrl(),request.getQuery());
    }

    @Override
    public String create(String longUrl) {
        return create(longUrl,null);
    }

    @Override
    public String create(String longUrl, Map<String, Object> query) {
        if (StringUtils.isBlank(longUrl)) {
            throw new ServiceException("长链不能为空");
        }
        if (!CollectionUtils.isEmpty(query)) {
            if (longUrl.contains("?")) {
                longUrl = longUrl + "&" + getQueryString(query);
            }else {
                longUrl = longUrl + "?" + getQueryString(query);
            }
        }

        List<String> shortUrls = create(CollectionUtil.newArrayList(longUrl));
        return shortUrls.get(0);
    }

    @Override
    public ShortLinkChannelEnum type() {
        return ShortLinkChannelEnum.Baidu;
    }

    private List<String> create(List<String> longUrls) {
        if (CollectionUtils.isEmpty(longUrls)) {
            throw new ShortUrlException("长链不能为空");
        }
        List<Map<String,String>> jsonArray = longUrls.stream()
                .map(longUrl -> {
                    Map<String,String> jsonObject = new HashMap<>(2);
                    jsonObject.put("LongUrl", longUrl);
                    jsonObject.put("TermOfValidity", TERM_OF_VALIDITY);
                    return jsonObject;
                })
                .collect(Collectors.toList());

        HEADERS.put("Content-Type",CONTENT_TYPE);

        BaiduShortLinkResponse response = HttpUtil.post(HOST, HEADERS, JsonUtils.toJsonString(jsonArray), BaiduShortLinkResponse.class);

        // 检查API响应是否成功
        if (response.getCode() != 0) {
            throw new ApiCallException("百度短链API调用失败: " + response.getErrorMsg());
        }
        return extractShortUrls((response.getShortUrls()));
    }

    private List<String> extractShortUrls(List<Dict> shortUrls) {
        List<String> result = new ArrayList<>();
        for (Dict shortUrlObj : shortUrls) {
            String shortUrl = shortUrlObj.getStr(SHORT_URL_KEY);
            result.add(shortUrl);
        }
        return result;
    }

}
