package org.example.common.exception;

import lombok.extern.slf4j.Slf4j;
import org.example.common.domain.R;
import org.example.common.enums.HttpCodeEnum;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author liutao
 * @since 2024/4/7
 */
@Slf4j
@RestControllerAdvice
public class ServiceExceptionHandler {


    @ExceptionHandler(Exception.class)
    public R<Object> handleException(Exception e) {
        log.error(e.getMessage(), e);
        return R.fail();
    }

    @ExceptionHandler(ServiceException.class)
    public R<Object> handleException(ServiceException e) {
        HttpCodeEnum httpCodeEnum = e.getHttpCodeEnum();
        if (Objects.nonNull(httpCodeEnum)) {
            return R.result(httpCodeEnum);
        }
        return R.fail(e.getMessage(),e.getData());
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public R<Object> methodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error(e.getMessage(),e);
        BindingResult bindingResult = e.getBindingResult();
        return R.fail(Arrays.toString(bindingResult.getAllErrors().stream().map(ObjectError::getDefaultMessage).toArray()));
    }

    @ExceptionHandler(BindException.class)
    public R<Object> handleBindException(BindException ex) {
        FieldError fieldError = ex.getBindingResult().getFieldError();
        if (Objects.nonNull(fieldError)) {
            log.error("参数校验异常:{}({})", fieldError.getDefaultMessage(),fieldError.getField());
            return R.fail(fieldError.getDefaultMessage());
        }
        return R.fail();
    }
    @ExceptionHandler(MissingServletRequestParameterException.class)
    public R<Object> missingServletRequestParameterException(MissingServletRequestParameterException e) {
        log.error(e.getMessage(),e);
        return R.result(HttpCodeEnum.PARAM_ERROR);
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public R<Object> constraintViolationException(ConstraintViolationException e) {
        log.error(e.getMessage(),e);
        String errorMessage = e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(","));
        return R.fail("参数校验失败:" + errorMessage);
    }

}
