package org.example.common.oapi;

import com.lark.oapi.Client;
import com.lark.oapi.core.enums.BaseUrlEnum;
import com.lark.oapi.core.httpclient.IHttpTransport;
import com.lark.oapi.core.request.FormData;
import com.lark.oapi.core.request.FormDataFile;
import com.lark.oapi.core.request.RawRequest;
import com.lark.oapi.core.response.RawResponse;
import com.lark.oapi.core.utils.IOs;
import com.lark.oapi.core.utils.OKHttps;
import com.lark.oapi.okhttp.*;
import org.apache.logging.log4j.util.Strings;
import org.example.common.util.JsonUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author liutao
 * @since 2024/9/4
 */
@Configuration
public class OApiConfig {

    @Bean
    @ConditionalOnProperty(name = "oapi.enable", havingValue = "true")
    public Client client(OApiProperties oApiProperties) {
        return Client.newBuilder(oApiProperties.getAppId(), oApiProperties.getAppSecret())
                .requestTimeout(30, TimeUnit.SECONDS)
                .openBaseUrl(BaseUrlEnum.FeiShu)
                .httpTransport(new FsHttpTransport(OKHttps.create(30, TimeUnit.SECONDS))).build();
    }

    @Bean
    @ConditionalOnProperty(name = "oapi.enable", havingValue = "true")
    public BiTableService biTableService(Client client) {
        return new BiTableServiceImpl(client);
    }

    static class FsHttpTransport implements IHttpTransport {

        private final OkHttpClient okHttpClient;

        public FsHttpTransport(OkHttpClient okHttpClient) {
            this.okHttpClient = okHttpClient;
        }
        private RequestBody buildReqBody(RawRequest request) {
            if (request != null && request.getBody() != null) {
                Object body = request.getBody();
                if (!(body instanceof FormData)) {

                    return RequestBody.create(MediaType.parse("application/json; charset=utf-8"), Objects.requireNonNull(JsonUtils.toJsonBytes(body)));
                } else {
                    String contentType = "multipart/form-data;charset=" + StandardCharsets.UTF_8;
                    MultipartBody.Builder builder = (new MultipartBody.Builder()).setType(Objects.requireNonNull(MediaType.parse(contentType)));
                    FormData formData = ((FormData) body);
                    for (Map.Entry<String, Object> entry : formData.getParams().entrySet()) {
                        builder.addFormDataPart(entry.getKey(), entry.getValue().toString());
                    }
                    for (FormDataFile file : formData.getFiles()) {
                        File finalFile = file.getFile();
                        builder.addFormDataPart(file.getFieldName(), Strings.isEmpty(file.getFileName()) ? "unknown" : file.getFileName(), RequestBody.create(MediaType.parse("application/octet-stream"), finalFile));
                    }
                    return builder.build();
                }
            } else {
                return null;
            }
        }

        @Override
        public RawResponse execute(RawRequest request) throws Exception {
            RequestBody body = this.buildReqBody(request);
            Request.Builder builder = (new Request.Builder()).url(request.getReqUrl()).method(request.getHttpMethod(), body);

            for (Map.Entry<String, List<String>> entry : request.getHeaders().entrySet()) {

                for (String o : entry.getValue()) {
                    builder.header(entry.getKey(), o);
                }
            }

            if (!(request.getBody() instanceof FormData)) {
                builder.header("content-type", "application/json; charset=utf-8");
            }

            Response response = this.okHttpClient.newCall(builder.build()).execute();
            RawResponse rawResponse = new RawResponse();
            rawResponse.setStatusCode(response.code());
            rawResponse.setHeaders(response.headers().toMultimap());
            if (request.isSupportDownLoad()) {
                assert response.body() != null;
                rawResponse.setBody(Objects.requireNonNull(IOs.readAll(response.body().byteStream())));
            } else {
                rawResponse.setBody(Objects.requireNonNull(response.body()).bytes());
            }

            return rawResponse;
        }
    }
}
