package org.example.common.trans;

import cn.hutool.core.lang.Dict;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.example.common.constant.Constants;
import org.example.common.exception.ServiceException;
import org.example.common.util.HttpUtil;
import org.example.common.util.JsonUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Service;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 讯飞翻译接口
 * @author liutao
 * @since 2024/10/16
 */
@Slf4j
@Service
@RequiredArgsConstructor
@ConditionalOnProperty(name = "translate.type", havingValue = "xfyun")
public class XfYunTranslationServiceImpl implements TranslationService{
    /**
     * OTS webapi 接口地址
     */
    private static final String WEB_ITS_URL = "https://itrans.xfyun.cn/v2/its";
    /**
     * 应用ID（到控制台获取）
     */
    @Value("${xf.appid:2f2c1455}")
    private String appId;
    @Value("${xf.apiKey:405796f886c313f8e1105fa563f6341d}")
    private String apiKey;
    @Value("${xf.apiSecret:ZjQ4MjM0ZDNmYTNhOGI2MzM3Y2UxOGNm}")
    private String apiSecret;




    @Override
    public String translate(TranslationRequest translationRequest) {
        try {
            String content = translationRequest.getContent();
            String from = translationRequest.getFrom();
            String to = translationRequest.getTo();
            if (StringUtils.isBlank(content)) {
                throw new ServiceException("翻译文本不能为空");
            }
            if (StringUtils.isBlank(from)) {
                from = Constants.DEFAULT_FROM;
            }
            if (StringUtils.isBlank(to)) {
                to = Constants.DEFAULT_TO;
            }
            // 组装http请求体
            String body = buildHttpBody(from, to, content);
            // 组装http请求头
            Map<String, String> header = buildHttpHeader(body);
            TranslationResponse response = HttpUtil.post(WEB_ITS_URL, header, body, TranslationResponse.class);
            if (response != null) {
                log.info("【ITS WebAPI 接口调用结果】\n{}", JsonUtils.toJsonString( response));
                Integer code = response.getCode();
                if (code != 0) {
                    throw new ServiceException("翻译失败");
                }
                TranslationResponse.TransData data = response.getData();
                return data.getResult().getTransResult().getDst();
            } else {
                throw new ServiceException("翻译失败");
            }
        } catch (Exception e) {
            throw new ServiceException("翻译失败");
        }
    }


    public Map<String, String> buildHttpHeader(String body) throws Exception {
        Map<String, String> header = new HashMap<String, String>();
        URL url = new URL(WEB_ITS_URL);

        //时间戳
        SimpleDateFormat format = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
        format.setTimeZone(TimeZone.getTimeZone("GMT"));
        Date dateD = new Date();
        String date = format.format(dateD);

        //对body进行sha256签名,生成digest头部，POST请求必须对body验证
        String digestBase64 = "SHA-256=" + signBody(body);

        //hmacsha256加密原始字符串
        String builder = "host: " + url.getHost() + "\n" +
                "date: " + date + "\n" +
                "POST " + url.getPath() + " HTTP/1.1" + "\n" +
                "digest: " + digestBase64;
        String sha = hmacsign(builder, apiSecret);

        //组装authorization
        String authorization = String.format("api_key=\"%s\", algorithm=\"%s\", headers=\"%s\", signature=\"%s\"", apiKey, "hmac-sha256", "host date request-line digest", sha);
        log.info("【ITS WebAPI authorization】\n{}", authorization);

        header.put("Authorization", authorization);
        header.put("Content-Type", "application/json");
        header.put("Accept", "application/json,version=1.0");
        header.put("Host", url.getHost());
        header.put("Date", date);
        header.put("Digest", digestBase64);
        System.out.println("【ITS WebAPI header】\n" + header);
        return header;
    }

    private static String signBody(String body) {
        MessageDigest messageDigest;
        String encodestr = "";
        try {
            messageDigest = MessageDigest.getInstance("SHA-256");
            messageDigest.update(body.getBytes(StandardCharsets.UTF_8));
            encodestr = Base64.getEncoder().encodeToString(messageDigest.digest());
        } catch (Exception e) {
            log.error(e.getMessage(),e);
        }
        return encodestr;
    }

    /**
     * hmacsha256加密
     */
    private static String hmacsign(String signature, String apiSecret) throws Exception {
        Charset charset = StandardCharsets.UTF_8;
        Mac mac = Mac.getInstance("hmacsha256");
        SecretKeySpec spec = new SecretKeySpec(apiSecret.getBytes(charset), "hmacsha256");
        mac.init(spec);
        byte[] hexDigits = mac.doFinal(signature.getBytes(charset));
        return Base64.getEncoder().encodeToString(hexDigits);
    }

    /**
     * 组装http请求体
     */
    public String buildHttpBody(String from, String to, String content) {
        Dict body = new Dict();
        Dict business = new Dict();
        Dict common = new Dict();
        Dict data = new Dict();
        //填充common
        common.put("app_id", appId);
        //填充business
        business.put("from", from);
        business.put("to", to);
        //填充data
        byte[] textByte = content.getBytes(StandardCharsets.UTF_8);
        String textBase64 = Base64.getEncoder().encodeToString(textByte);
        data.put("text", textBase64);
        //填充body
        body.put("common", common);
        body.put("business", business);
        body.put("data", data);
        return JsonUtils.toJsonString( body);
    }
}
